import subprocess
import time
from pathlib import Path
import sys
import getpass
import platform

try:
    import jwt # pip install pyjwt
except ImportError:
    print("Please install jwt library (pip install pyjwt or pip3 install pyjwt)")
    sys.exit()

try:
    from cryptography.hazmat.primitives import serialization # pip install cryptography 
    from cryptography.hazmat.backends import default_backend
except ImportError:
    print("Please install cryptography library (pip install cryptography or pip3 install cryptography)")
    sys.exit()

# To generate PKI Key pair:
# ssh-keygen -t rsa -b 4096 -m PEM -f <Key Path>.key
# openssl rsa -in <Key Path>.key -pubout -outform PEM -out <Key Path>.key.pub

while True:
    private_key_path = input("Enter private key file path: ")

    if (len(private_key_path.strip()) > 0):
        break
    else:
        print("Please enter a valid file path for the private key\n")

public_key_path = private_key_path + ".pub"

private_key_file = Path(private_key_path)
if (not private_key_file.is_file()):
    subprocess.run(["ssh-keygen" , "-t", "rsa", "-b", "4096", "-m", "PEM", "-f", private_key_path])
    subprocess.run(["openssl", "rsa", "-in", private_key_path, "-pubout", "-outform", "PEM", "-out", public_key_path])

try:
    private_key = open(private_key_path, 'r').read()
except Exception as error:
    print(error)
    print("\nFailed to create private key. Please check to see that you entered a valid path. \nIf you are using a passphrase, please make sure to create a passcode with at least 5 characters.")
    sys.exit()

exp_input = input("Enter expiration time in days (defaults to 1 day from now): ")

expiration_time = 86400 if exp_input == "" else ( 86400 * int(exp_input) )

iss_name = input("Name of public key on FileMaker Server Admin Console: ")

while True:
    if (len(iss_name.strip()) > 0):
        iss_name = iss_name.strip()
        break
    else:
        print("Please enter a name with at least 1 non-space character\n")
        iss_name = input("Name of public key on FileMaker Server Admin Console: ")

payload_data = {
    "iss": iss_name,
    "aud": "fmsadminapi",
    "exp": int(time.time()) + expiration_time #86400 is 24 hours (1 day)
}

password = None
password_provided = getpass.getpass("Enter passphrase (empty for no passphrase):")
if (not password_provided == ""):
    password = password_provided.encode()

try:
    key = serialization.load_pem_private_key(private_key.encode(), password, default_backend())
except Exception as error:
    print(error)
    sys.exit()

token = jwt.encode(
    payload=payload_data,
    key=key,
    algorithm='RS256'
)

token_file_path = Path(private_key_path).parent.joinpath(iss_name.replace(" ", "_") + ".txt")

if (platform.system() == 'Linux'):
    token_file = open(token_file_path, "wb")
    try:
        token_file.write(bytes(token, 'UTF-8'))  
    except:
        token_file.write(token)
else:
    token_file = open(token_file_path, "w")
    token_file.write(token)


token_file.close()


print("Token has been successfully written to " + iss_name + ".txt")
